#!/bin/bash
declare -r SCRIPT_NAME=$(basename $0 .sh)

# where are the mdAddr files located?
if [ "${MDENV_ADDROBJ_PATH:+DEFINED}" == 'DEFINED' ]; then
    # if the environment variable is set, just use that
    declare -r mdAddrPath="${MDENV_ADDROBJ_PATH}"
else
    ###EDIT_THIS##  use a hard coded default path specific to the local environment
    declare -r mdAddrPath="The Path To Your AddrObj Directory"
fi
function errexit ()
{
    echo ""  1>&2
    echo "${SCRIPT_NAME}: ERROR: ${*}" 1>&2
    exit 1
}

# standard warning message function
function warn ()
{
    echo ""  1>&2
    echo "${SCRIPT_NAME}: Warning: ${*}"  1>&2
}

# define the conditions a valid file must meet
function is_valid_file ()
{
    if [ ! "${1}" ]; then
        warn "null name given to is_valid_file"
        return 0
    fi

    if [ ! -e "${1}" ]; then errexit "does not exist: ${1}"; fi

    if [ ! -f "${1}" ]; then errexit "is not a regular file: ${1}"; fi

    if [ ! -r "${1}" ]; then errexit "lacks read permission: ${1}"; fi

    return 0
}


# define the conditions a valid executable file must meet
function is_valid_exec ()
{
    is_valid_file "${1}"

    if [ ! -x "${1}" ]; then errexit "lacks execute permission: ${1}"; fi

    return 0
}

# define the conditions a valid directory must meet
function is_dir_readable ()
{
    if [ ! "${1}" ]; then
        warn "null name given to is_dir_readable"
        return 0
    fi

    if [ ! -e "${1}" ]; then errexit "does not exist: ${1}"; fi

    if [ ! -d "${1}" ]; then errexit "is not a directory: ${1}"; fi

    if [ ! -r "${1}" ]; then errexit "lacks read permission: ${1}"; fi

    if [ ! -x "${1}" ]; then errexit "lacks traverse permission: ${1}"; fi

    return 0
}

declare -r JWRAP_JAR='mdAddr.jar'
is_valid_file "${JWRAP_JAR}"

# make sure the given AddrObj directory contains the required items
is_valid_file "${mdAddrPath}/libmdAddr.so"

# we need to find the right Java environment to use, it is a little tricky

# First, if we do have a JAVA_HOME environment variable defined, we use that
if [ "${JAVA_HOME:+DEFINED}" == 'DEFINED' ]; then
    is_dir_readable "${JAVA_HOME}/bin"
    declare -r JAVAC_CMD="${JAVA_HOME}/bin/javac"
    is_valid_exec "${JAVAC_CMD}"
    declare -r JAVA_CMD="${JAVA_HOME}/bin/java"
    is_valid_exec "${JAVA_CMD}"
else
    # see if there is a 'javac' anywhere in the path set
    if tmp=$(which javac)
    then
        declare -r JAVAC_CMD="${tmp}"
        is_valid_exec "${JAVAC_CMD}"
    else
        errexit "no instance of the 'javac' command could be found"
    fi

    # see if there is a 'java' anywhere in the path set
    if tmp=$(which java)
    then
        declare -r JAVA_CMD="${tmp}"
        is_valid_exec "${JAVA_CMD}"
    else
        errexit "no instance of the 'java' command could be found"
    fi
fi

export LD_LIBRARY_PATH="${mdAddrPath}"
echo ""
if [ -f AddressSample.class ]; then
    echo "Warning: AddressSample.class already present, using that one"
else
    echo "Compiling Address Object sample"
    if ! "${JAVAC_CMD}" -classpath "${JWRAP_JAR}" AddressSample.java
    then
        errexit "unable to compile AddressSample.java"
    fi
fi
echo ""
"${JAVA_CMD}" "-Djava.library.path=.:${mdAddrPath}" -classpath ".:${JWRAP_JAR}" AddressSample 

echo ""
if [ -f ParseZipStreetSample.class ]; then
    echo "Warning: ParseZipStreetSample.class already present, using that one"
else
    echo "Compiling Parse ZipStreet sample"
    if ! "${JAVAC_CMD}" -classpath "${JWRAP_JAR}" ParseZipStreetSample.java
    then
        errexit "unable to compile ParseZipStreetSample.java"
    fi
fi
echo ""
"${JAVA_CMD}" "-Djava.library.path=.:${mdAddrPath}" -classpath ".:${JWRAP_JAR}" ParseZipStreetSample 
echo ""

exit 0

